/*
** A simple Morse code program.
** Copyright (C), Ralph Holland, 1990.
** mailto:vk1brh@arising.com.au
**
** Description:
** I developed this program to train for the Australian Amateur Radio
** Morse code examinations. It was originally written for an Amiga and
** subsequently ported to the IBM pc.
** On an IBM it is best run under DOS, but I have sucessfully run it
** under a DOS prompt in Windows and Windows 95, Windows 98, Windows 200
** and recently on Windows XP. No further development has been done on
** this program since Nov 1990 - which was when I passed my Morse exam
** and was awarded my Unrestricted licence.
**
** Future:
** Encode all symbols and have a switch to disable punctuation.
** Have a switch to disable prosigns
** Migrate codebase to Visual Studio C++
**
** History:
** 009: 01 Jan 2004 - RBH, Updated my email address
** 008: 20 Aug 1990 - RBH, Allow any ASCII file to processed as random words.
** 007: 17 Aug 1990 - RBH, Added a Dictionary for random words.
** 006: 27 Jun 1990 - RBH, Changed the default pitch to 500Hz.
** 005: 12 Jun 1990 - RBH, Made -e work with concat (^) symbols.
** 004: 12 Jun 1990 - RBH, added -l and -n suboptions
** 003: 28 May 1990 - RBH, rewrote for IBM pc
** 002: 27 May 1990 - RBH, fixed by removing extra delay in Pound.
** 001: 26 May 1990 - RBH, modified to use Sine wave data
**                         Should change the envelope amplitude just after keying
**                         amd before keying off to remove clicks!
** 000: 25 May 1990 - Ralph Holland, Original writing
**
*/


#ifdef __TURBOC__         /* Change if using Borland compiler or Visual C++ */
#include <stdlib.h>       /* standard funcs */
#include <string.h>
#include <io.h>           /* open etc. */
#include <fcntl.h>        /* file constants */
#include <sys/stat.h>     /* mode constants */
#include <time.h>         /* time defs */
#include <ctype.h>        /* char macro defs */
#define  BIT_MAX_RAND 15  /* Number of bits for max rand in TURBO */

#else

/* Amiga standard library stuff */
#include <exec/types.h>
#include <exec/memory.h>
#include <exec/devices.h>
#include <hardware/custom.h>
#include <hardware/dmabits.h>
#include <libraries/dos.h>
#include <devices/audio.h>
#include <devices/timer.h>
#include libraries/dos.h>
#define  BIT_MAX_RAND 16

#endif

#include <stdio.h>

#define WordFileName  "C:/lib/words"  /* my need to fix path for compiler */
#define DefaultDotDelay           30
static  unsigned long DotDelay  = DefaultDotDelay;
static  unsigned long LetterDelay = DefaultDotDelay*3;
static  unsigned long WordDelay   = DefaultDotDelay*7;
static  double Frequency = 500.0;
static  char   SendBuffer[128];
static  short  Echo         = 0;
static  short  RandomGroups = 0;
static  short  RandomWords  = 0;
static  short  GroupSize    = 0;
static  short  DoNumbers    = 0;
static  short  DoLetters    = 0;
static  short  Volume       = 2;
static char *SelectionTable =
  "0123456789abcdefghijklmnopqrstuvwxyz.,?";
static  char   *pNumbers;
static  char   *pLetters;
static  unsigned long WordFileSize;
static  char    TempFileName[128];

#define SizeofSendBuffer sizeof(SendBuffer);

void   Error(char *);              /* Reports and terminates */
void   Help();
void   Send( char * );             /* Send ascii text */
short  Pound( char * );            /* Send code symbols */
void   Dit();                      /* Send dot */
void   Dah();                      /* Send dash */
short  ControlC();                 /* Control C check */ 
void   InitSelectWord();
void   SelectWord( char * SendBuffer, int WordFd );

/* machine dependent routines */
void   sound( unsigned );          /* Enable the sound channels */
void   nosound();                  /* Disable the sound channels */
void   delay( unsigned long );     /* delay a specified time */
void   Terminate();                     /* Tidies up */
short  SetVolume(char *);          /* Establish a volume for the sound */

   /* Note hd to compile with non-ANSII C compiler hence no protos */
void main( argc, argv )
  int argc;
  char *argv[];
{

  char *FileName;
  FILE *InputFile;
  int  WordFd;
  int  Speed;
  int  Pitch;
  int  Gap;
  int  GapUnit;
  int  i;
  int  index;
  int  CWords;
  unsigned long LongIndex;
  int  LineLength = 0;

  pNumbers = SelectionTable;
  pLetters = SelectionTable + 10;
  argv++;
  FileName = NULL;
  Speed    = 15;      /* 15 words per minute - default */
  Pitch    = 800;  /* 1kHz */
  Gap      = 0;
  InitSelectWord();
  while( --argc ) {
    switch( *argv[0] ) {
      case '-':
        switch( toupper( argv[0][1] ) ) {
          case 'S':
            Speed = atoi( &(argv[0][2]) );
            break;

          case 'G':
            Gap = atoi( &(argv[0][2]) );
            break;

          case 'P':
            Pitch = atoi( &(argv[0][2]) );
            break;
           
          case 'N':
            DoNumbers = 1;
            break;
              
          case 'L':
            DoLetters = 1;
            break;

          case 'E':  
            Echo = 1;
	    LineLength = 0;
            break;
           
          case 'R':
            RandomGroups = 1;
            GroupSize = atoi( &(argv[0][2]) );
            if ( GroupSize <= 0 ) GroupSize = 5;
            break;
           
          case 'W':
            RandomWords = 1;
            GroupSize = atoi( &(argv[0][2]) );
            if ( GroupSize <= 0 ) GroupSize = 5;
            break;
            
          case 'H':
            Help();
            break;
           
          case 'V':
            Volume = SetVolume( &argv[0][2] );
            break;

          case 'Q':
            Terminate();
           
        }
        break;

      case '?':
        Help();
       break;

      default:
        FileName = argv[0];
        break;
    }
    argv++;
  }
  /* Validate these args */
  if (RandomWords) {
    /* Now open the word file name */
    if ( FileName==NULL ) {
      strcpy( TempFileName, WordFileName );
    }
    else {
      strcpy( TempFileName, FileName );
    }
    WordFd = open( TempFileName, O_RDONLY, S_IFREG );
    if ( WordFd < 0 ) {
      if ( strcmp( WordFileName, TempFileName ) == 0 ) {
        WordFd = open( "WORDS", O_RDONLY, S_IFREG );
      }
    }
    if ( WordFd < 0 ) {
      printf( "Unable to open the dictionary file %s\n", TempFileName );
      Terminate();
    }
    FileName = NULL;   /* This is now a word file */
    if (RandomGroups) {
      Help();
    }
    WordFileSize = filelength( WordFd );
  }
  /*
  ** Now Establish the frequency and speed paramaters
  */
  if (Pitch > 0 ) {
    Frequency = Pitch;
  }
  if (Speed > 0 ) {
    /* DotDelay = 25000 / (24 * Speed);  */
    DotDelay = 1200 / Speed;
    if (Gap == 0) {
      Gap = Speed;  /* standard gap wpm */
    }
    /* GapUnit = 25000 / ( 24 * Gap ); */
    GapUnit = 1200 / Gap;
    LetterDelay = GapUnit * 3;
    WordDelay = GapUnit * 7;
  }
     
  printf( "CW speed %d w.p.m at a frequency of %6d Hz\n", Speed, Pitch );
  printf( "with letter gap speed = %d \n", Gap );
  delay( DotDelay );   /* My pc doesn't delay properly unless I do this */
  if (FileName != NULL) {
    if( (InputFile=fopen( FileName, "r" )) == NULL ) {
      printf("Unable to open %s\n", FileName );
    }
    else {
      while( fgets( SendBuffer, sizeof(SendBuffer), InputFile ) ) {
        Send( SendBuffer );
      }
    }
  }
  else {
    if (RandomGroups || RandomWords) {
      /* seed the random number generator */
      randomize();
      CWords = 0;
      for ( ; ; ) {
        if (RandomWords) {
	  SelectWord( SendBuffer, WordFd );
	  Send( SendBuffer );
	  if (Echo) {
	    LineLength += strlen( SendBuffer );
	    if ( 
              ((++CWords >= GroupSize) && (GroupSize > 1) ) || 
              (LineLength >= 64) 
            ) {
	      printf( "\n" );
	      LineLength = 0;
              CWords = 0;
	    }
	  }
	}
	else {
	  for ( i = 0; i < 5 ; i++ ) {
	    index = rand();
	    if (DoLetters) {
	      index = index % 26;
	      SendBuffer[ i ] = pLetters[ index ];
	    }
	    else if (DoNumbers) {
	      index = index % 10;
	      SendBuffer[ i ] = pNumbers[ index ];
	    }
	    else {
	      index = index % strlen(SelectionTable);
	      SendBuffer[ i ] = SelectionTable[ index ];
	    }
	  }
	  SendBuffer[ 5 ] = ' ';
	  SendBuffer[ 6 ] = '\0';
	  Send( SendBuffer );
	}
      }
    }
    else {
      Send( "^CT CW program by VK1BRH, Ralph Holland. ^AR " );
    }
  }
  Terminate();
}
#define UIOrdWords 20
#define UISector 512

static int   IRandWords;
static char *OrdWords[ UIOrdWords ];
static char *RandWords[ UIOrdWords ];
static char Scratch[ UISector ];

void
InitSelectWord()
{
  int I;
  for ( I = 0; I < UIOrdWords; I++ ) {
    OrdWords[ I ] = NULL;
    RandWords[ I ] = NULL;
  }
  IRandWords = 0;
}

void
SelectWord( char * SendBuffer, int WordFd )
{
  int Done;
  unsigned long LongIndex;
  char *NewLine;
  char *Space;
  char *p;
  char *End;
  int Found = 0;
  int IOrdWords, I;
  
  if ( (IRandWords >= GroupSize /* UIOrdWords */) 
    || (RandWords[ IRandWords ] == NULL) ) 
  {
    InitSelectWord();
    IRandWords = 0;
    IOrdWords = 0;
    LongIndex = (long)rand() | ((long)(rand())<<BIT_MAX_RAND);
    lseek( WordFd, LongIndex % WordFileSize, SEEK_SET );
    Found = read( WordFd, Scratch, sizeof( Scratch ) );
    /* Now locate words */
    End = p + strlen( Scratch );
    for ( p = Scratch; !isspace( *p ); p++ );
    do {
      for ( ;isspace( *p ); p++ );
      OrdWords[ IOrdWords++ ] = p;
      for ( ; !isspace( *p ); p++ );
      *p++ = '\0';
    } while( 
     (p < End) && (IOrdWords < GroupSize) && (IOrdWords < UIOrdWords) );
    /* Now randomize them */
    for ( IRandWords = 0; IRandWords < IOrdWords; ) {
      I = rand() % IOrdWords;
      if ( (OrdWords[ I ] != NULL) && (*OrdWords[ I ] != '\0') ) {
        RandWords[ IRandWords++ ] = OrdWords[ I ];
        OrdWords[ I ] = NULL;
      }
    }
    IRandWords = 0;
  }
  strcpy( SendBuffer, RandWords[ IRandWords++ ] );
  strcat( SendBuffer, " " );     
}

void 
Help()
{
  printf( "morse [filename] [switches]\n" );
  printf( "      -s... sets speed to w.p.m\n" );
  printf( "      -g... Sets letter gap as ... w.p.m.\n" );
  printf( "      -e    Echos chars sent\n" );
  printf( "      -p... sets frequency in Hz\n" );
  printf( "      -rnn  sends random groups groupsize nn (default 5)\n" );
  printf( "      -wnn  send random words group size nn (default 5)\n" );
  printf( "            this option works with any ascii file.\n" );
  printf( "      -l    Letters only in random groups\n" );
  printf( "      -n    Numbers and letters in random groups\n" );
  printf( "      -v... Volume (max 64 doesn't work on IBM)\n" );
  printf( "      -q    Quit\n" );
  printf( "\n" );
}


static char *Alpha[] = {
  ".-"          /* a */,
  "-..."        /* b */,
  "-.-."        /* c */,
  "-.."         /* d */,
  "."           /* e */,
  "..-."        /* f */,
  "--."         /* g */,
  "...."        /* h */,
  ".."          /* i */,
  ".---"        /* j */,
  "-.-"         /* k */,
  ".-.."        /* l */,
  "--"          /* m */,
  "-."          /* n */,
  "---"         /* o */,
  ".--."        /* p */,
  "--.-"        /* q */,
  "._."         /* r */,
  "..."         /* s */,
  "-"           /* t */,
  "..-"         /* u */,
  "...-"        /* v */,
  ".--"         /* w */,
  "-..-"        /* x */,
  "-.--"        /* y */,
  "--.."        /* z */
};
#define AlphaBase 'a'

static char *Number[] = {
  "-----"       /* 0 */,
  ".----"       /* 1 */,
  "..---"       /* 2 */,
  "...--"       /* 3 */,
  "....-"       /* 4 */,
  "....."       /* 5 */,
  "-...."       /* 6 */,
  "--..."       /* 7 */,
  "---.."       /* 8 */,
  "----."       /* 9 */
};
#define NumberBase '0'

void
Send( Text )
char *Text;
{
  short Gap = 0;
  char  *p;
  int   c;
  char  Concat[128];
 
  p = Text;
  if ( (p == NULL) || (*p == '\0') ) {
    return;
  }
  for ( c = *p++; c != '\0'; c = *p++ ) {
    if (Echo) {
      switch (c) {
	case '\r': 
	case '\n':
	  printf("\n" );
	  break;
	default:
	  printf( "%c", toupper( c ) );
	  break;
      }
    }
    if ( isalpha(c) ) {
      if ( isupper(c) ) {
        c = tolower(c);
      }
      Gap = Pound( Alpha[ c - AlphaBase] );
    }
    else if ( isdigit(c) ) {
      Gap = Pound( Number[ c - NumberBase ] );
    }
    else {
      switch( c ) {
 
        /* Space symbols */
        case ' ':
        case '\t':
        case '\r':
        case '\n':
           delay( WordDelay );
           Gap = 0;
           break;
 
        /* Combination symbols */
        case '^':
          Concat[0] = '\0';
          for( c = *p++; (c != ' ') && (c != '\0') ; c = *p++ ) {
            if ( isdigit(c) ) {
              strcat( Concat, Number[ c - NumberBase ] );
            }
            else if ( isalpha( c ) ) {
              c = tolower( c );
              strcat( Concat, Alpha[ c - AlphaBase ] );
            }
            else {
              continue;
            }
            if (Echo) {
              printf( "%c", c );
            }
          }           
          if (Echo) {
            printf( " " );
          }
          (void)Pound( Concat );
          delay( LetterDelay - DotDelay );
          Gap = 0;
          break;

/*
** NOTE: specials should probably be put in an array too!
** not all special chars are defined 
*/

        /* Specials */
        case '.':
           Gap = Pound( "._._._" );
           break;
 
        case ',':
           Gap = Pound( "--..--" );
           break;
            
        case '?':
           Gap = Pound( "..--.." );
           break;
 
        case ':':
           Gap = Pound( "---..." );
           break;
 
        case ';':
           Gap = Pound( "-.-.-." );
           break;
 
        case '(':
           Gap = Pound( "-.--.-" );
           break;
 
        case '/':
           Gap = Pound( "-..-." );
           break;
 
        default:
           Gap = 0;
           break;
      }
    }
    if (Gap) {
      Gap = 0;
      delay( LetterDelay - DotDelay );
    }
    if (ControlC()) {
      Error("bye");
    }
  }
}

/* Send the code symbols */
short
Pound( Code )
char * Code;
{
  char *p;
  int   c;
  short Gap = 0;

  p = Code;
  if ( (p == NULL) || (*p == '\0') ) {
    return 0;
  }
  for ( c = *p++; c != '\0'; c = *p++ ) {
    switch( c ) {
      case '.':
        Dit();
        break;

      case '-':
      case '_':
        Dah();
        break;

      default:
        printf("%c is an invalid symbol in code %s\n", c, Code );
        return 0;
    }
    Gap = 1;
  }
  return Gap;
}

  /* send a dot */
void Dit()
{
  sound( Frequency );
  delay( DotDelay );
  nosound();
  delay( DotDelay );
}

  /* send a dash */
void Dah()
{
  sound( Frequency );
  delay( 3 * DotDelay );
  nosound();
  delay( DotDelay );
}



/*
** Error reporting routine
*/
void Error(message)
  char *message;
{
  printf("%s\n", message);
  Terminate();
}


/*
** Machine Dependent code follows, Note on the IBM, these routines
** exist
*/
#ifndef __TURBOC__

#define LEFT0B  0
#define RIGHT0B 1
#define RIGHT1B 2
#define LEFT1B  3
#define LEFT0F  1
#define RIGHT0F 2
#define RIGHT1F 4
#define LEFT1F  8

#define WAVELENGTH 16
#define CLOCK 3579545
#define MAXVOLUME 64
#define SOUNDPREC -40

extern struct MsgPort *CreatePort();
extern struct AudChannel aud[];
extern UWORD dmacon;

void sound( Frequency )
  unsigned Frequency;
{

  static UBYTE channels;
  struct AudChannel *leftRegs, *rightRegs;
  static short soundOpenned = 0;

  if (!soundOpenned) {

    soundOpenned = 1;
    /* allocate I/O Blocks from chip Public */
    if (((allocIOB = (struct IOAudio *)AllocMem(sizeof(struct IOAudio),
       MEMF_PUBLIC | MEMF_CLEAR)) == 0) ||
       ((lockIOB = (struct IOAudio *)AllocMem(sizeof(struct IOAudio),
       MEMF_PUBLIC | MEMF_CLEAR)) == 0)) {
       Error( "Out of Memory" );
    }

    if (OpenDevice(AUDIONAME, 0, allocIOB, 0 ) != 0) {
      Error("Cannot open audio device");
    }
    device = allocIOB->ioa_Request.io_Device;

    allocIOB->ioa_Request.io_Message.mn_Node.ln_Pri = SOUNDPREC;
    if ((port = CreatePort("sound example", 0)) == 0) {
      Error("Cannot create message port" );
    }
    allocIOB->ioa_Request.io_Message.mn_ReplyPort = port;
    allocIOB->ioa_Request.io_Command = ADCMD_ALLOCATE;
  
    allocIOB->ioa_Request.io_Flags = ADIOF_NOWAIT;
    allocIOB->ioa_Data = allocationMap;
    allocIOB->ioa_Length = sizeof(allocationMap);

    /* allocate the channels now */
    BeginIO(allocIOB);
    if (WaitIO(allocIOB)) {
      Error("Channel allocation failed");
    }
    /* Initialize the IO block to lock channels */
    lockIOB->ioa_Request.io_Message.mn_ReplyPort = port;
    lockIOB->ioa_Request.io_Device = device;
    lockIOB->ioa_Request.io_Unit = allocIOB->ioa_Request.io_Unit;
    lockIOB->ioa_Request.io_Command = ADCMD_LOCK;
    lockIOB->ioa_AllocKey = allocIOB->ioa_AllocKey;

    /* lock the channel */
    SendIO(lockIOB);
    if (CheckIO(lockIOB)) {
      Error( "Channel stolen" );
    }
    channels = (ULONG)(allocIOB->ioa_Request.io_Unit);
    leftRegs = (channels & LEFT0F) ? &aud[LEFT0B] : &aud[LEFT1B];
    rightRegs = (channels & RIGHT0F) ? &aud[RIGHT0B] : &aud[RIGHT1B];

    /* allocate waveform memory from chip ram */
    if ((WaveData = (BYTE *)AllocMem( WAVELENGTH, MEMF_CHIP))==0) {
      Error( "out of memory" );
    }
    /*
    WaveData[0] = 127;
    WaveData[1] = -127;
    */
    WaveData[0] = 0;
    WaveData[1] = 49;
    WaveData[2] = 90;
    WaveData[3] = 117;
    WaveData[4] = 127;
    WaveData[5] = 117;
    WaveData[6] = 90;
    WaveData[7] = 49;
    WaveData[8] = 0;
    WaveData[9] = -49;
    WaveData[10] = -90;
    WaveData[11] = -117;
    WaveData[12] = -127;
    WaveData[13] = -117;
    WaveData[14] = -90;
    WaveData[15] = -49;
    leftRegs->ac_ptr = (UWORD *)WaveData;
    rightRegs->ac_ptr = (UWORD *)WaveData;
    leftRegs->ac_len = WAVELENGTH / 2;
    rightRegs->ac_len = WAVELENGTH / 2;

    leftRegs->ac_per = CLOCK / Frequency / WAVELENGTH;
    rightRegs->ac_per = CLOCK / Frequency / WAVELENGTH;
        
    leftRegs->ac_vol = Volume;
    rightRegs->ac_vol = Volume;

  }
  dmacon = DMAF_SETCLR | channels << DMAB_AUD0;
  
}

/* Turn the sound off */
nosound()
{
  if (allocIOB) {
    allocIOB->ioa_Request.io_Command = ADCMD_FINISH;
    allocIOB->ioa_Request.io_Flags   = ADIOF_SYNCCYCLE;
    DoIO(allocIOB);
  }
}

static TimerOpen = 0;
static struct timerequest timermsg;
static struct MsgPort     *TimerPort;
UBYTE allocationMap[] = {
 LEFT0F | RIGHT0F,
 LEFT0F | RIGHT1F,
 LEFT1F | RIGHT0F,
 LEFT1F | RIGHT1F
};
struct IOAudio *allocIOB = NULL;
struct IOAudio *lockIOB  = NULL;
struct Device  *device   = NULL;
struct MsgPort *port     = NULL;
BYTE *WaveData           = NULL;


void delay( milliseconds )
  unsigned long milliseconds;
{
  unsigned long seconds;
  unsigned long microseconds;

  microseconds = milliseconds * 1000;
  if (!TimerOpen) {
    if ((TimerPort = CreatePort("Timer Port", 0)) == NULL) {
      Error( "Unable to open timer port" );
    }
    if ((OpenDevice( "timer.device", UNIT_VBLANK,&timermsg, 0 )) != NULL) {
      Error( "Unable to open timer" );
    }
    TimerOpen = 1;
    timermsg. tr_node . io_Message . mn_ReplyPort = TimerPort ;
    timermsg. tr_node . io_Command = TR_ADDREQUEST ;
    timermsg. tr_node . io_Flags = 0 ;
    timermsg. tr_node . io_Error = 0 ;
  }
  if (microseconds >= 1000000) {
    seconds = microseconds / 1000000;
    microseconds = microseconds % 1000000;
  }
  else {
    seconds = 0;
  }
  timermsg. tr_|ime.tv_secs = seconds;
  timermsg. tr_time.tv_micro = microseconds;
  DoIO(&timermsg);

}

struct AmigaDate {
  long days;
  long minutes;
  long ticks;
};

/*
** This function exists in Turbo C
*/
void randomize()
{
  struct AmigaDate Date;
  DateStamp( &Date );
  srand( Date.ticks + Date.minutes );
}

/*
** Terminate routine 
*/
void Terminate()
{
  if (WaveData != 0) {
    FreeMem(WaveData, WAVELENGTH);
  }
  if (port != 0) {
    DeletePort(port);
  }
  if (device != 0) {
    CloseDevice(allocIOB);
  }
  if (lockIOB != 0) {
    FreeMem(lockIOB, sizeof(struct IOAudio));
  }
  if (allocIOB != 0) {
    FreeMem( allocIOB, sizeof(struct IOAudio));
  }
  if (TimerOpen) {
    CloseDevice(&timermsg);
    TimerOpen = 0;
  }
  exit(0);
}

short SetVolume( VolString )
  char *VolString;
{
  short Volume;
  
  Volume = atoi( VolString );
  if (Volume > 64) {
    Volume = 64;
  }
  else if (Volume <= 0) {
    Volume = 1;
  }
  return Volume;
}

short ControlC() 
{
  if ( SetSignal(0,0) & SIGBREAKF_CTRL_C ) {
    SetSignal(0,0xffffffff);
    return 1;
  }
  return 0;
}

#else

/* Do not need to do much on the IBM pc */
void Terminate()
{
  exit(0);
}


/*
** Code to handle control-C on the IBM - not necessary on some compilers
*/

static short HandlerInstalled = 0;
static short ControlCTyped = 0;

void MyHandler( void )
{
  ControlCTyped = 1;
}

/* 
** See if control-C typed
*/
#include <conio.h>

short ControlC()
{
  if (!HandlerInstalled) {
    ctrlbrk( MyHandler );
  }
  (void)kbhit();
  if ( ControlCTyped ) {
    ControlCTyped = 0;
    return 1;
  }
  return 0;
}

short SetVolume( char *VolString )
{
  printf( "Not implemented for the IBM pc\n" );
  return 0;
}

#endif

