//==============================================================================
// Description:
// This implements Cyclic Redundancy Check sums - based on polynomials
// Two classical polynomials are defined by:
//    CRC< unsigned int, 16, 0x1021 >       CRC16
//    CRC< unsigned long, 32, 0x04c11db7 >  CRC32
// Refer to  C++ Report Vol 7/No 2 Feb 1995, page 8 for background.
//
// History:
// 000: 22 Feb 95, RBH - Initial writing
//
//==============================================================================

#ifndef CRC_H
#define CRC_H

template<class T,int bits,long polynomial> class Crc {
public:
   enum { crcHiBit = 1 << (bits-1) };
   enum { crcMask = ((crcHiBit-1) << 1)|1 };
   Crc() {
      T crcPoly = polynomial;
         // populate the generation table
      table[0]=0;
      for (int d = 0; d < charBits; ++d) {
         int diff = 1 << d;
         for (int j=0; j < diff; ++j) {
            table[j+diff] = table[j] ^ crcPoly;
         }
         bool highBit = (crcPoly & crcHiBit) != 0;
         crcPoly = (crcPoly << 1 ) & crcMask;
         if (highBit) {
            crcPoly ^= polynomial;
         }
      }
   }

      // Hash functions
   T Hash( const char *buffer, size_t size, T initial=-1 ) {
      unsigned const char *pData = (unsigned const char *)buffer;
      T h = initial;
      unsigned char c;
      for (size_t i = size; i > 0; i-- ) {
         c = *pData++;
         register unsigned char index = (unsigned char)(h >> shiftAmount) ^ c;
         h = table[index] ^ ((h << charBits) & crcMask);
      }
      return h;
   }
   T Hash( const char *buffer, T initial=-1 ) {
      unsigned const char *pData = (unsigned const char *)buffer;
      T h = initial;
      unsigned char c;
      while ((c= *pData++)!=0) {
         register unsigned char index = (unsigned char)(h >> shiftAmount) ^ c;
         h = table[index] ^ ((h << charBits) & crcMask);
      }
      return h;
   }
private:
   enum { charBits = 8 };                              // # bits in a char
   enum { tableSize = 1 << charBits };                 // size of generate table
   enum { charMask = (1 << charBits)-1 };              // all bits on in a character
   enum { shiftAmount = bits - charBits };

   T table[tableSize];                         // the generator table.
};


#endif

